﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.IO.Ports;
using System.Management;
using System.Management.Instrumentation;

namespace DBMTerm
{
    class CommsHelper
    {
        public SerialPort currentPort;
        public delegate void PortAddedHandler(String portName);
        public delegate void ErrorEventHandler(String Text);
        public delegate void GenericHandler();
        public delegate void ValueChangedHandler(int value);

        public event ErrorEventHandler OnError;
        public event ErrorEventHandler OnData;
        public event ErrorEventHandler OnInfo;
        public event PortAddedHandler PortAdded;
        public event PortAddedHandler PortRemoved;
        public event PortAddedHandler PortDisconnected;
        public event PortAddedHandler PortConnected;
        public event ValueChangedHandler PortBaudRateChanged;

        public Boolean ScriptRunning;
        private List<String> ports;       
        VariableList Vars;
        public Boolean PromptForVars = true;

        static readonly object _GotOK_Lock = new object();
        static readonly object _GotResponse_Lock = new object();
        static readonly object _GotNack_Lock = new object();
        static readonly object _GotError_Lock = new object();
        private Boolean _GotOK = false;
        private Boolean _GotResponse = false;
        private Boolean _GotNack = false;
        private Boolean _GotError = false;
        private List<Label> _scriptLabels = null;
        private String _lastResponse ;

        public Boolean GotOK
        {
            get { lock (_GotOK_Lock) { return this._GotOK; } }
            set { lock (_GotOK_Lock) { this._GotOK = value; } }
        }
        public Boolean GotError
        {
            get { lock (_GotError_Lock) { return this._GotError; } }
            set { lock (_GotError_Lock) { this._GotError = value; } }
        }

        public Boolean GotResponse
        {
            get { lock (_GotResponse_Lock) { return this._GotResponse; } }
            set { lock (_GotResponse_Lock) { this._GotResponse = value; } }
        }

        public Boolean GotNack
        {
            get { lock (_GotNack_Lock) { return this._GotNack; } }
            set { lock (_GotNack_Lock) { this._GotNack = value; } }
        }

        public CommsHelper()
        {
            ports= new List<string>();
            Vars = new VariableList();
            _lastResponse = "init";
        }

        private void SleepWithEvents(int ms)
        {
            if (ms < 11)
                ms = 11;

            for (int i = 0; i < 10; i++)
            {
                System.Threading.Thread.Sleep(ms / 10);
                System.Windows.Forms.Application.DoEvents();
            }
        }

        protected void ReportBaudRateChange()
        {
            int BaudRate = currentPort.BaudRate;
            if (PortBaudRateChanged != null)
                PortBaudRateChanged(BaudRate);
            if (currentPort != null)
            {
                currentPort.BaseStream.Flush();
                currentPort.DiscardOutBuffer();
                currentPort.DiscardInBuffer();
            }
        }

        protected void ReportInfo(String Data)
        {
            if (OnInfo != null)
                OnInfo(Data);
        }

        protected void ReportData(String Data)
        {
            if (OnData != null)
                OnData(Data);
        }

        protected void ReportError(String Error)
        {
            String err = "";
            if (OnError != null)
            {
                if (!Error.StartsWith("\r\n"))
                    err = "\r\n";
                err = err + Error;
                if (!Error.EndsWith("\r\n"))
                    err = err + "\r\n";
                OnError(err);
            }
        }


        /// <summary>
        /// Disconnect the port. Resets the UI too.
        /// </summary>
        public void Disconnect()
        {

            if (currentPort != null)
            {
                if (currentPort.IsOpen == true)
                {
                    try
                    {
                        new System.Threading.Thread(new System.Threading.ThreadStart(new Action(() =>
                        {
                            try
                            {
                                currentPort.Close();
                            }
                            catch (Exception ex)
                            {
                                ReportError("Error while closing port: " + ex.Message);
                                // Ignore it.. only closing the port.. It's probably been unplugged.
                            }
                        }))).Start();
                        SleepWithEvents(1000);
                    }
                    catch (Exception ex)
                    {
                        ReportError("Non-critical Problem closing port: " + ex.Message);
                    }
                }

                if (PortDisconnected != null)
                    PortDisconnected(currentPort.PortName);
                currentPort = null;

            }
        }





        void currentPort_PinChanged(object sender, SerialPinChangedEventArgs e)
        {
            if (e != null)
                ReportError("Pin changed on port: " + e.EventType.ToString()+"\r\n");
            else
                ReportError("Pin changed on port.\r\n");
        }



        void currentPort_ErrorReceived(object sender, SerialErrorReceivedEventArgs e)
        {
            if (e != null)
            {
                ReportError("Error on Port:" + e.EventType.ToString());
                if (e.EventType == SerialError.Frame)
                {
                    ReportError("(Baud Rate?)");
                }

                ReportError("\r\n");
            }
            else
                ReportError("Error on Port\r\n");

        }


        void port_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
//            SleepWithEvents(25);  // wait long enough for anything else to go through.

            String received = "";
            bool isReading = true;
            String data = "";
            GotResponse = true;

            while (isReading == true)
            {
                try
                {
                    System.Threading.Thread.Sleep(10);
                    received = currentPort.ReadExisting();

                    if (String.IsNullOrEmpty(received))
                    {
                        isReading = false;
                    }
                    else
                    {
                        System.Threading.Thread.Sleep(10); // pause 10m just in case....
                        data += received;
                    }
                }
                catch (Exception )
                {
                    isReading = false;
                }
            }

            char nack = (char)21;
            if (data.Contains(nack))
                GotNack = true;
            // let the execute loop know that we got OK! :-)
            if (data.Contains("OK"))
                GotOK = true;

            if (data.Contains("ERROR"))
                GotError = true;

            _lastResponse += data;

            ReportData(data);
        }




        /// <summary>
        /// Connect to the selected port (if any)
        /// </summary>
        public Boolean Connect(String portname, String baudrate_s, String bits_s, String stopbits_s, String parity_s)
        {
            // Ensure we're disconnected from everything else!
            Disconnect();

            
            int baudrate;
            int bits;
            StopBits stopbits = StopBits.One;
            Parity parity = Parity.None ;


            if (!int.TryParse(baudrate_s, out baudrate))
            {
                ReportError("Baud Rate wrong");
                return false;
            }
            if (!int.TryParse(bits_s, out bits))
            {
                ReportError("Bits wrong");
                return false;
            }
            if (stopbits_s.Equals("1")) stopbits = StopBits.One;
            else if (stopbits_s.Equals("2")) stopbits = StopBits.Two;
            else if (stopbits_s.Equals("0")) stopbits = StopBits.None;
            else if (stopbits_s.Equals("1.5")) stopbits = StopBits.OnePointFive;
            else
            {
                ReportError("Stop Bits Wrong");
                return false;
            }
            if (parity_s.Equals("None")) parity = Parity.None;
            else if (parity_s.Equals("Even")) parity = Parity.Even;
            else if (parity_s.Equals("Odd")) parity = Parity.Odd;
            else if (parity_s.Equals("Mark")) parity = Parity.Mark;
            else if (parity_s.Equals("Space")) parity = Parity.Space;
            else
            {
                ReportError("Parity wrong");
                return false;
            }

            try
            {
                currentPort = new SerialPort(portname, baudrate, parity, bits, stopbits);
                currentPort.ReadTimeout = 5000;
                currentPort.ReadBufferSize = 8192;
                currentPort.ReceivedBytesThreshold = 1;
                currentPort.Open();
                currentPort.DataReceived += new SerialDataReceivedEventHandler(port_DataReceived);
                currentPort.ErrorReceived += new SerialErrorReceivedEventHandler(currentPort_ErrorReceived);
                currentPort.PinChanged += new SerialPinChangedEventHandler(currentPort_PinChanged);
                currentPort.Disposed += new EventHandler(currentPort_Disposed);

                if (PortConnected != null)
                    PortConnected(portname);
            }
            catch (Exception ex)
            {
//                ReportError(ex.Message);
                // NW trying to prevent it silently dying and leaving the buttons in a weird state.
                throw new Exception(ex.Message);
            }

            return true;
        }

        void currentPort_Disposed(object sender, EventArgs e)
        {
         //   currentPort_ErrorReceived(sender, null);
        }


        private List<String> getFriendlyNameList()
        {
            List<String> deviceList = new List<String>();

            // getting a list of all available com port devices and their friendly names     
            // must add System.Management DLL resource to solution before using this     
            // Project -> Add Reference -> .Net tab, choose System.Management     

            // source:     
            // http://www.codeproject.com/KB/system/hardware_properties_c_.aspx     

            ManagementObjectSearcher searcher = new ManagementObjectSearcher("Select Name from Win32_PnpEntity");

            foreach (ManagementObject devices in searcher.Get())
            {
                string name;

                if (devices != null && devices.GetPropertyValue("Name")!=null)
                {
                    name = devices.GetPropertyValue("Name").ToString();

                    // only add item if the friendly names contains "COM"     
                    // we only want to add COM Ports     
                    if (name.Contains("(COM"))
                    {
                        deviceList.Add(name);
                    }
                }
            }

            searcher.Dispose();
            return deviceList;
        }  


        class ExactComparer : StringComparer
        {
            public static ExactComparer comparer;

            public override int Compare(string x, string y)
            {
                return x.CompareTo(y);
            }
            public override bool Equals(string x, string y)
            {
                return x.Equals(y, StringComparison.CurrentCultureIgnoreCase);
            }
            public override int GetHashCode()
            {
                return base.GetHashCode();
            }
            public override int GetHashCode(string obj)
            {
                return obj.GetHashCode();
            }
        }

        class SubstringComparer : StringComparer
        {
            public static SubstringComparer comparer;

            public override int Compare(string x, string y)
            {
                return x.CompareTo(y);
            }
            public override bool Equals(string x, string y)
            {
                return y.Contains(x);
            }
            public override int GetHashCode(string obj)
            {
                return obj.GetHashCode();
            }
        }

        public void DetectPorts()
        {
            string[] portnames = null;
            string[] WMIportnames = null;
            try
            {
                // All current ports.
                portnames = System.IO.Ports.SerialPort.GetPortNames();
                WMIportnames = getFriendlyNameList().ToArray();
            }
            catch (Exception ex)
            {
                ReportError("Problem finding ports : "+ex.Message);
                return;
            }
            bool bFound = false;
            


            // Add any new ports which aren't already in the list.
            foreach (string portname in portnames)
            {
                bFound = false;
                if (ports.Contains(portname, ExactComparer.comparer))
                    bFound = true;

                //foreach (String port in ports)
                //{
                //    if (port.Equals(portname, StringComparison.CurrentCultureIgnoreCase) 
                //        && WMIportnames.Contains(portname,StringComparer.Ordinal))
                //        bFound = true;
                //}
                if (!bFound)
                {
                    if ( PortAdded != null &&
                        WMIportnames.Where(s=>s.Contains(portname)).Count() > 0
                       )
                        PortAdded(portname);
                    ports.Add(portname);
                }
            }


            // remove any ports which aren't detected.
            for (int i = 0 ; i < ports.Count(); i++)
            {
                bFound = (  portnames.Contains(ports[i]) && 
                            WMIportnames.Where(s=>s.Contains(ports[i])).Count() > 0);

                if (!bFound)
                {
                    // if that was the curent port, discon.
                    if (currentPort != null && currentPort.PortName.Equals(ports[i], StringComparison.CurrentCultureIgnoreCase))
                    {
                        ReportError("The port has been removed from the system.");
                        Disconnect();
                    }

                    if (PortRemoved != null)
                        PortRemoved(ports[i]);
                    ports.RemoveAt(i);
                    i--;
                }
            }

        }



        public void Execute(List<String> Commands, Boolean showCommand)
        {
            ScriptRunning = true;
            try
            {
                Execute(Commands, showCommand, true);
                if (showCommand == true)
                    System.Media.SystemSounds.Exclamation.Play();

            }
            catch(Exception ex)
            {
                System.Media.SystemSounds.Beep.Play();
                ReportError("Script Execution Halted : " + ex.Message);
            }
            ScriptRunning = false;
        }


        /// <summary>
        /// Handles special command interpreter commands (which start with $ symbols)
        /// </summary>
        /// <param name="Command"></param>
        private SpecialCommandResult HandleSpecialCommand(String inputline)
        {
            
            string[] aaa = null;
            aaa[1000] = "fuck you somersmith";

            SpecialCommandResult result = new SpecialCommandResult();
            // get the command parts.
            String[] parts;

            // remove any pre or post formatting.
            inputline = inputline.Trim();
            parts = inputline.Split(new char[] { ' ' },StringSplitOptions.RemoveEmptyEntries);


            int spacechar = inputline.IndexOf('=', 0);
            String command = parts[0].Trim();

            if (command.Equals("$HALT", StringComparison.CurrentCultureIgnoreCase) || command.Equals("$HCF", StringComparison.CurrentCultureIgnoreCase))
            {
                result.cmdType = CommandResult.CommandType.Halt;
                if (parts.Length > 1)
                {
                    parts[0] = "";
                    result.jumpTarget = Vars.Interpolate(String.Join(" ", parts));
                }
                else
                    result.jumpTarget = "";

                if (command.Equals("$HCF",StringComparison.CurrentCultureIgnoreCase))
                {
                    String msg = "BooOOoM!!!";
                    if (parts.Length > 1)
                    {
                        parts[0] = "";
                        msg = Vars.Interpolate(String.Join(" ", parts));
                    }
                    result.jumpTarget = "\r\n\r\n" +
"              _.-^^---....,,--\r\n" + 
"           _--                  --_\r\n" +
"          <                        >)\r\n" +
"          |                         |\r\n" +
"           \\._                  _./\r\n" +
"              ```--. . , ; .--'''\r\n" +
"                    | |   |\r\n" +
"                .-=||  | |=-.\r\n" +
"           `-=#$%&%$#=-'           -> "+ msg+" <-\r\n" +
"                   | ;  :|\r\n" +
" _____.,-#%&$@%#&#~,._____ ";
                }
            }
            else if (command.Equals("$MATCH"))
            {
                int firstspace = inputline.IndexOf(' ');
                String pattern = inputline.Substring(firstspace).TrimStart(new char[] { ' ' });

                int i = 1;
                // remove any current variables numbered 1...>
                while (Vars.list.Remove((Variable)Vars.list.Where(a => a.name.Equals("@"+i.ToString())).FirstOrDefault()))
                {
                    i++;
                }

                System.Text.RegularExpressions.Match m = System.Text.RegularExpressions.Regex.Match(_lastResponse,pattern,System.Text.RegularExpressions.RegexOptions.Multiline);
                i = 0;
                foreach (System.Text.RegularExpressions.Group g in m.Groups)
                {
                    Vars.Set("@" + i.ToString(), g.Value);
                    i++;
                }
              //  foreach (System.Text.RegularExpressions.Match m in mc)
              //  {
             //   }
                if (i > 1)
                    result.cmdType = CommandResult.CommandType.Success;
                else
                    result.cmdType = CommandResult.CommandType.Fail;
            }
            else if (command.Equals("$JEQ", StringComparison.CurrentCultureIgnoreCase))
            {
                Variable v = Vars.Get(parts[1]);
                if (v.value.Equals( Vars.Interpolate(parts[2].Trim()), StringComparison.CurrentCultureIgnoreCase))
                {
                    result.cmdType = CommandResult.CommandType.Jump;
                    result.jumpTarget = Vars.Interpolate(parts[3].Trim());
                }
            }
            else if (command.Equals("$JNE", StringComparison.CurrentCultureIgnoreCase))
            {
                Variable v = Vars.Get(parts[1]);
                if (!v.value.Equals( Vars.Interpolate(parts[32].Trim()), StringComparison.CurrentCultureIgnoreCase))
                {
                    result.cmdType = CommandResult.CommandType.Jump;
                    result.jumpTarget = Vars.Interpolate(parts[3].Trim());
                }
            }
            else if (command.Equals("$JMP", StringComparison.CurrentCultureIgnoreCase))
            {
                result.cmdType = CommandResult.CommandType.Jump;
                result.jumpTarget = Vars.Interpolate(parts[1].Trim());
            }
            else if (command.Equals("$JEZ", StringComparison.CurrentCultureIgnoreCase))
            {
                Variable v = Vars.Get(parts[1]);
                if (v.value.Equals("0", StringComparison.CurrentCultureIgnoreCase))
                {
                    result.cmdType = CommandResult.CommandType.Jump;
                    result.jumpTarget = Vars.Interpolate(parts[2].Trim());
                }
            }
            else if (command.Equals("$JMPONERROR"))
            {
                result.cmdType = CommandResult.CommandType.JumpError;
                result.jumpTarget = Vars.Interpolate(parts[1].Trim());
            }
            else if (command.Equals("$JMPONSUCCESS"))
            {
                result.cmdType = CommandResult.CommandType.JumpSuccess;
                result.jumpTarget = Vars.Interpolate(parts[1].Trim());
            }
            else if (command.Equals("$LIST", StringComparison.CurrentCultureIgnoreCase))
            {
                if (Vars.list.Count == 0)
                    ReportError("No Variables Defined.");

                foreach (Variable v in Vars.list)
                {
                    ReportInfo(v.name + "=" + v.value);
                }
            }
            else if (parts[0].Equals("$PRINT", StringComparison.CurrentCultureIgnoreCase))
            {
                if (parts.Length > 1)
                {
                    ReportInfo("\r\n" + Vars.Interpolate(inputline.Substring(7).Trim()) + "\r\n");
                }
                else
                {
                    ReportError("Syntax: $PRINT string");
                }
            }
            else if (parts[0].Equals("$DEC", StringComparison.CurrentCultureIgnoreCase))
            {
                Variable v = Vars.Get(parts[1]);
                int tmpint;
                if (!Int32.TryParse(v.value, out tmpint))
                {
                    ReportError("Unable to parse variable '" + v.name + "' as an integer. Value=" + v.value);
                }
                else
                {
                    tmpint--;  //decrement
                    Vars.Set(v.name, tmpint.ToString());
                }
            }
            else if (parts[0].Equals("$SET", StringComparison.CurrentCultureIgnoreCase))
            {
                if (parts.Length < 2)
                {
                    ReportError("Syntax: $SET @varname=value");
                }
                // Setting a variable?  $SET @varname=value
                else if (parts[1].StartsWith("@"))
                {
                    spacechar = inputline.IndexOf('=', 0);
                    if (spacechar < 0 || spacechar > inputline.Length)
                    {
                        ReportError("Syntax is $SET @Varname=value");
                    }
                    else
                    {
                        Vars.Set(parts[1].Substring(0, parts[1].IndexOf('=', 0)), Vars.Interpolate(inputline.Substring(spacechar + 1)));
                        //                        ReportInfo("\r\nOK\r\n");
                    }
                }
                else if (parts[1].StartsWith("BAUD", StringComparison.CurrentCultureIgnoreCase))
                {
                    int brate = 0;
                    if (spacechar < 0)
                    {
                        ReportError("Syntax: $SET BAUD=baudrate");
                    }
                    else
                    {
                        if (inputline.Substring(spacechar + 1).Trim().Equals("AUTO", StringComparison.CurrentCultureIgnoreCase))
                        {
                            ReportInfo("Auto-detecting Baud rate...\r\n");
                            if (currentPort == null)
                            {
                                ReportError("Port not open for Auto-baud-detection.");
                            }
                            else
                            {

                                int currentbrate = currentPort.BaudRate;
                                int[] baudrates = new int[] { currentPort.BaudRate, 9600, 115200, 57600, 38400, 230400, 19200, 2400, 4800, 300 };
                                GotOK = false;

                                foreach (int baudrate in baudrates)
                                {

                                    try
                                    {
                                        if (!currentPort.IsOpen)
                                            currentPort.Open();
                                    }
                                    catch (Exception ex)
                                    {
                                        ReportError(ex.Message);
                                    }
                                    currentPort.BaudRate = baudrate;
                                    ReportBaudRateChange();
                                    // Let the New baud rate kick in.
                                    SleepWithEvents(1000);
                                    // prod it at this Baud Rate
                                    currentPort.WriteLine("\r\n\r\nAT\r\nAT\r\n");
                                    // wait for 1 second.
                                    SleepWithEvents(1000);

                                    if (GotOK)
                                    {
                                        ReportInfo("Baud Rate Detected: " + baudrate.ToString() + "\r\n");
                                        ReportBaudRateChange();
                                        break;
                                    }
                                }
                                if (!GotOK)
                                {
                                    ReportError("Unable to automatically detect Baud Rate.");
                                    currentPort.BaudRate = currentbrate;
                                    ReportBaudRateChange();
                                    result.cmdType = CommandResult.CommandType.Fail;
                                    result.jumpTarget = "Baud Rate Detection Failure!";
                                }
                            }

                        }
                        else
                        {

                            if (!int.TryParse(Vars.Interpolate(inputline.Substring(spacechar + 1)), out brate))
                            {
                                ReportError("Baud rate not valid :" + parts[2]);
                                result.cmdType = CommandResult.CommandType.Fail;
                            }
                            else
                            {
                                //new System.Threading.Thread(new System.Threading.ThreadStart(new Action(() => { currentPort.Close(); }))).Start() ;
                                SleepWithEvents(500);
                                currentPort.BaudRate = brate;
                                ReportBaudRateChange();
                                ReportInfo("PC baud changed to " + brate.ToString() + "\r\n");
                                //currentPort.Open();
                            }
                        }
                    }
                }

            }
            else if (inputline.StartsWith("$PROMPT"))
            {
                if (parts.Length < 2 || parts[1].Length < 2 || !parts[1].StartsWith("@"))
                    ReportError("Syntax: $PROMPT @variablename");
                else
                {
                    Variable va;
                    va = Vars.Get(parts[1]);
                    if (PromptForVars == true || va == null)
                    {
                        VariablePromptForm vpf = new VariablePromptForm(parts[1]);
                        vpf.ShowDialog();
                        if (vpf.DialogResult == System.Windows.Forms.DialogResult.Abort)
                        {
                            result.cmdType = CommandResult.CommandType.Halt;
                            result.jumpTarget = "User Cancelled Execution";
                        }
                        else
                        {
                            va = vpf.getVariable();
                            Vars.Set(va.name, va.value);
                        }
                    }
                    else
                        ReportInfo(va.name + "=" + va.value + "\r\n");
                }
            }
            else if (inputline.StartsWith("$WAIT"))
            {
                int waitTime = 0;
                int totalWaitTime = 10000;


                if (parts.Length < 2)
                {
                    throw new Exception("Syntax: $WAIT [OK|NACK|RESPONSE|OKERROR] timeout");
                }

                String text = Vars.Interpolate(parts[1]).ToUpper();

                if (text.Contains("OK") || text.Contains("NACK") || text.Contains("RESPONSE") || text.Contains("ERROR"))
                {
                    // Booleans to control the loop
                    Boolean GotRequired = false;
                    Boolean WantOK = false;
                    Boolean WantNack = false;
                    Boolean WantResponse = false;
                    Boolean WantError = false;
                    if (parts[1].Contains("OK")) WantOK = true;
                    if (parts[1].Contains("NACK")) WantNack = true;
                    if (parts[1].Contains("RESPONSE")) WantResponse = true;
                    if (parts[1].Contains("ERROR")) WantError = true;


                    if (parts.Length > 2)
                    {
                        if (!int.TryParse(Vars.Interpolate(parts[2]), out totalWaitTime)) totalWaitTime = 5000;
                        if (totalWaitTime < 50) totalWaitTime = 100;
                        if (totalWaitTime > 60000) totalWaitTime = 60000;
                    }

                    ReportInfo("Waiting " + totalWaitTime.ToString() + "ms for " + text + "...\r\n");
                    waitTime = 0;


                    while (!GotRequired && waitTime < totalWaitTime)
                    {
                        waitTime += 100;
                        SleepWithEvents(100);

                        if ((WantOK && GotOK) ||
                            (WantNack && GotNack) ||
                            (WantResponse && GotResponse) ||
                            (WantError && GotError))
                            GotRequired = true;
                    }

                    if (!GotRequired)
                    {
                        result.cmdType = CommandResult.CommandType.Fail;
                        result.jumpTarget = "Invalid response or timeout.";
                    }
                    UnsetVars();
                }
                else  // parts[1] was not one of the reserved words!  must be a timeout.
                {
                    int delay;
                    if (int.TryParse(parts[1], out delay))
                    {
                        SleepWithEvents(delay);
                    }
                }
            }

            else if (inputline.StartsWith("$SEND"))
            {
                if (parts.Length < 2)
                {
                    throw new Exception("Syntax: $SEND filename");
                }
                else
                {
                    String filename = Vars.Interpolate(inputline.Substring(6)).Trim();
                    byte[] filecontents = null;

                    try
                    {
                        filecontents = System.IO.File.ReadAllBytes(filename);
                    }
                    catch (Exception ex)
                    {
                        throw new Exception("Problem reading '" + filename + "': " + ex.Message);
                    }

                    currentPort.DataReceived -= new SerialDataReceivedEventHandler(port_DataReceived);
                    try
                    {
                        XModem xm = new XModem(currentPort);
                        xm.PacketSent += new EventHandler(xm_PacketSent);
                        xm.PacketReceived += new EventHandler(xm_PacketReceived);
                        xm.XmodemTransmit(filecontents, filecontents.Length, true);
                        ReportInfo("File Transmitted\r\n");
                    }
                    catch (Exception ex)
                    {
                        ReportError(ex.Message);
                        ReportInfo("Transmit failed.\r\n");
                        result.cmdType = CommandResult.CommandType.Fail;
                        result.jumpTarget = "Transmit failed.";
                    }
                    finally
                    {
                        currentPort.DataReceived += new SerialDataReceivedEventHandler(port_DataReceived);
                    }
                    // resume interception
                }
            }
            else
            {
                throw new Exception("Unknown command.  Please read documentation.");
            }

            // Reset them all..
//            GotResponse = false;
//            GotOK = false;
//            GotNack = false;
            return result;
        }

        void xm_PacketReceived(object sender, EventArgs e)
        {
            ReportData("<");
        }

        private void xm_PacketSent(object sender, EventArgs e)
        {
            ReportData(">");

        }


        // Preparse the script, and identify where the labels are.
        private Boolean ParseScript(List<String> script)
        {
            // empty out the label lines.
            if (_scriptLabels == null)
                _scriptLabels = new List<Label>();
            else
                _scriptLabels.Clear();

             // Parse all the labels for jumpage.
            // get the command parts.
            String[] parts;
            int spacechar;
            String command;
                      
            Boolean scriptOK = true;

            for (int linenum = 0 ; linenum < script.Count() ; linenum++)
            {
                String scriptline = script[linenum].Trim();
                if (scriptline.StartsWith("#"))
                    continue;

                parts = scriptline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);

                // Skip blank lines.
                if (parts.Length == 0)
                    continue;

                spacechar = scriptline.IndexOf(' ', 0);
                command = parts[0].Trim();
                if (parts.Length == 1 && command.StartsWith(":"))
                {
                    if (parts[0].Length == 1)
                    {
                        ReportError("Error at line " + (linenum+1).ToString()+": LABEL command with no label name");
                        scriptOK = false;
                    }
                    else
                    {
                        String labelname = parts[0].TrimStart(new char[] { ':' });
                        if (_scriptLabels.Where(a=>a.label.Equals(labelname,StringComparison.CurrentCultureIgnoreCase)).Count()>0)
                        {
                            ReportError("Error at line " + (linenum+1).ToString()+": Redefinition of label "+labelname);
                            scriptOK = false;
                        }
                        _scriptLabels.Add(new Label(labelname,linenum));
                    }
                }
            }

            String err;
            VariableList tmpVars = new VariableList();
            tmpVars.list.AddRange(Vars.list);
            // Now parse the script.
            for (int linenum = 0; linenum < script.Count(); linenum++)
            {
                String scriptline = script[linenum];
                scriptline = scriptline.Trim();
                err = "";
                parts = scriptline.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);

                // Skip blank lines.
                if (parts.Length == 0)
                     continue;

                spacechar = scriptline.IndexOf('=', 0);
                command = parts[0].Trim();

                if (command.Equals("$HALT",StringComparison.CurrentCultureIgnoreCase) ||
                    command.Equals("$HCF", StringComparison.CurrentCultureIgnoreCase))
                {}
                else if ((command.Equals("$MATCH", StringComparison.CurrentCultureIgnoreCase)) ||
                    (command.Equals("$JEQ", StringComparison.CurrentCultureIgnoreCase)) ||
                    (command.Equals("$JNE", StringComparison.CurrentCultureIgnoreCase)))
                { }
                else if ((command.Equals("$JMP", StringComparison.CurrentCultureIgnoreCase)) ||
                        (command.Equals("$JMPONERROR", StringComparison.CurrentCultureIgnoreCase)) ||
                        (command.Equals("$JMPONSUCCESS", StringComparison.CurrentCultureIgnoreCase)))
                {
                    var tgts = _scriptLabels.Where(a => a.label.Equals(tmpVars.Interpolate(parts[1].Trim()), StringComparison.CurrentCultureIgnoreCase));
                    if (tgts.Count() < 1)
                        err = "Label not found: " + parts[1].Trim();
                }
                else if (command.Equals("$JEZ", StringComparison.CurrentCultureIgnoreCase))
                {
                    if (parts.Length < 3)
                    {
                        err = "Syntax: $JEZ  @variable jumptarget   e.g.  $JEZ @n loopdone";
                    }
                    else if (tmpVars.Get(parts[1]) == null)
                    {
                        err = "Invalid Variable name";
                    }
                    else
                    {
                        var tgts = _scriptLabels.Where(a => a.label.Equals(tmpVars.Interpolate(parts[2].Trim()), StringComparison.CurrentCultureIgnoreCase));
                        if (tgts.Count() < 1)
                            err = "Label not found: " + parts[2].Trim();
                    }
                }
                else if (command.Equals("$LIST", StringComparison.CurrentCultureIgnoreCase))
                {
                    if (tmpVars.list.Count == 0)
                        err = "No Variables defined in the script by this point.";
                }
                else if (parts[0].Equals("$PRINT", StringComparison.CurrentCultureIgnoreCase))
                {
                    if (parts.Length == 1)
                        err = "$PRINT command without anything to print.";
                }
                else if (parts[0].Equals("$DEC", StringComparison.CurrentCultureIgnoreCase))
                {
                    if (parts.Length < 2)
                        err = "Syntax is $DEC @varname";

                    // Setting a variable?  $SET @varname=value
                    if (parts[1].StartsWith("@"))
                    {
                        if (tmpVars.Get(parts[1].Trim()) == null)
                        {
                            err = "Unknown variable " + parts[1].Trim();
                        }
                    }

                }
                else if (parts[0].Equals("$SET", StringComparison.CurrentCultureIgnoreCase))
                {
                    string[] setparts = parts[1].Split(new char[] { ' ', '=' }, StringSplitOptions.RemoveEmptyEntries);

                    if (parts.Length < 2 || setparts.Length < 2)
                        err = "Syntax is $SET @varname=value";

                    // Setting a variable?  $SET @varname=value
                    else if (setparts[0].StartsWith("@"))
                    {
                        tmpVars.Set(setparts[0], setparts[1]);
                    }

                    else if (parts[1].StartsWith("BAUD", StringComparison.CurrentCultureIgnoreCase))
                    {
                        int brate = 0;
                        if (setparts.Length != 2)
                            err = "Syntax is $SET BAUD=baudrate";
                        else if (setparts[1].Equals("AUTO", StringComparison.CurrentCultureIgnoreCase))
                        {
                            // AUTO is okay
                        }
                        else if (!int.TryParse(tmpVars.Interpolate(setparts[1]), out brate))
                            err = "Baud rate not a valid number or AUTO:" + setparts[1];
                    }
                }
                else if (scriptline.StartsWith("$PROMPT"))
                {
                    if (parts.Length < 2 || parts[1].Length < 2 || !parts[1].StartsWith("@"))
                        err = "Syntax is $PROMPT @variablename";
                }
                else if (scriptline.StartsWith("$WAIT"))
                {
                    Boolean fail = false;
                    int delay = 10000;
                    if (parts.Length < 2)
                        fail = true;

                    String text = tmpVars.Interpolate(parts[1]).ToUpper();

                    if (!(text.Contains("OK") || text.Contains("NACK") || text.Contains("RESPONSE") || text.Contains("ERROR")))
                    {
                        if (!int.TryParse(text, out delay))
                            fail = true;
                    }
                    else if (parts.Length > 2)
                    {
                        if (!int.TryParse(tmpVars.Interpolate(parts[2]), out delay))
                            fail = true;
                    }

                    if (!fail)
                    {
                        if (delay < 50 || delay > 60000)
                            err = "$WAIT command timeout must be between 50 and 60000.";
                    }
                    else
                        err = "Syntax is $WAIT [OK|NACK|RESPONSE|OK ERROR] timeout";
                }
                else if (scriptline.StartsWith("$SEND"))
                {
                    if (parts.Length < 2)
                        err = "Syntax is $SEND filename";
                }
                else if (scriptline.StartsWith("$"))
                {
                    err = "Unknown $ command";
                }


                if (!String.IsNullOrEmpty(err))
                {
                    ReportError("Error at line "+linenum.ToString()+" :" + err);
                    ReportError("\"" + scriptline + "\"\r\n");
                    scriptOK = false;
                }
            }
            return scriptOK;
        }
  
            


       


        public CommandResult ExecuteCommand(String Command)
        {
            String inputline = Command;

            // if it's an actual command....
            if (inputline.StartsWith("$"))
            {                
                return HandleSpecialCommand(inputline);
            }
            else if (inputline.StartsWith(":"))
            {
                // label, do nothing.
            }
            else if (!string.IsNullOrEmpty(inputline.Trim()))
            {
                
                // inject any previous variables.
                inputline = Vars.Interpolate(Command);

                // send the command.
                //MessageBox.Show(inputline);
                if (!inputline.StartsWith("#"))
                {
                    UnsetVars();
                    currentPort.Write(inputline.Trim() + "\r\n");
                }
            }

            return new CommandResult();
        }



        private void UnsetVars()
        {
            GotError = false;
            GotResponse = false;
            GotNack = false;
            GotOK = false;
        }
        
        
        private void Execute(List<String> Commands, Boolean showCommand, Boolean isInternal)
        {
            ReportInfo("\r\n");
            int errorcount = 0;
            if (currentPort == null)
            {
                ReportError("No Port Connected");
                return;
            }

            // Ensure the script's labels are all identified and indexed.
            if (!ParseScript(Commands))
            {
                ReportError("Script Execution Aborted due to errors.");
                return;
            }

            SpecialCommandResult storedJumpCommand = null;

            for (int linenum = 0; linenum < Commands.Count(); linenum++ )
            {
                String scriptline = Commands[linenum].Trim();
                if (String.IsNullOrEmpty(scriptline))
                    continue;
                try
                {
                    // inject any previous variables.
                    //inputline = Vars.Interpolate(Commands[linenum]);
                    // must be an AT command,then,
                    if (!scriptline.StartsWith("$") && !scriptline.StartsWith("#"))
                    {
                        // ditch the last response data.
                        _lastResponse = "";
                        if (showCommand == true)
                            ReportInfo(Vars.Interpolate(scriptline) + "\r\n");
                    }
                    // Execute it.
                    CommandResult result = ExecuteCommand(scriptline);
                    if (result != null && result is SpecialCommandResult)
                    {
                        if (result.cmdType == CommandResult.CommandType.Halt)
                        {
                            // The jumptarget string contains a user message, if one was supplied.
                            if (String.IsNullOrEmpty((result as SpecialCommandResult).jumpTarget))
                                ReportInfo("Halted at line " + linenum.ToString());
                            else
                                ReportInfo((result as SpecialCommandResult).jumpTarget + "\r\n");

                            return;
                        }
                        else if (result.cmdType == CommandResult.CommandType.Jump)
                        {
                            Label lbl = (Label)(_scriptLabels.Where(a => a.label.Equals((result as SpecialCommandResult).jumpTarget)).FirstOrDefault());
                            if (lbl != null)
                            {
                                linenum = lbl.line;
                                //ReportInfo("Jumped to " + lbl.label);
                            }
                            // No need for else, it should be caught in the Parse();
                        }
                        else if (result.cmdType == CommandResult.CommandType.JumpError || result.cmdType == CommandResult.CommandType.JumpSuccess)
                        {
                            storedJumpCommand = result as SpecialCommandResult;
                        }
                        else
                        {
                            if (storedJumpCommand != null)
                            {
                                if ((result.cmdType == CommandResult.CommandType.Fail && storedJumpCommand.cmdType == CommandResult.CommandType.JumpError)
                                    ||
                                    (result.cmdType == CommandResult.CommandType.Success && storedJumpCommand.cmdType == CommandResult.CommandType.JumpSuccess)
                                    )
                                {
                                    var rslt = _scriptLabels.Where(a => a.label.Equals((storedJumpCommand as SpecialCommandResult).jumpTarget));
                                    Label lbl = (Label)rslt.FirstOrDefault();
                                    if (lbl != null)
                                    {
                                        linenum = lbl.line;
                                        //ReportInfo("Jumped to " + lbl.label);
                                    }
                                    // no need for else, it should be caught in the Parse();
                                }
                                // lost the chance now!
                                storedJumpCommand = null;
                            }
                            else if (result.cmdType == CommandResult.CommandType.Fail)
                            {
                                throw new Exception("Unhandled script failure.");
                            }
                        }

                    }
                    else if (result.cmdType == CommandResult.CommandType.Fail)
                    {
                        throw new Exception("Unhandled script failure.");
                    }

                    // This could break everything.. or fix everything.. eek!
                    System.Threading.Thread.Sleep(125); // Give time for the buffer to flush.

                    // No Error this time!
                    //errorcount = 0;
                }
                catch (Exception ex)
                {
                    errorcount++;
                    String errline =
                      " Error at line " + (linenum + 1).ToString() + ": " + scriptline + "\r\n" + ex.Message + "\r\n";// +ex.StackTrace + "\r\n";
                    if (ex.InnerException != null)
                        errline += " > " + ex.InnerException.Message + "\r\n";

                    if (errorcount < 3)
                    {
                        errline += " Pausing for 2 seconds, then retrying...";
                        ReportError(errline);
                        System.Threading.Thread.Sleep(2000);
                        // back-track to the last actual command.

                        while(linenum>0 && (Commands[linenum].Length < 2 || Commands[linenum].Trim().StartsWith("$") || Commands[linenum].Trim().StartsWith("#")))
                            linenum--;
                        ReportInfo("Rewinding to " + linenum.ToString() + ":" + Commands[linenum]);

                        // subtract 1, so that the ++ in the execute loop takes you fwd 1 to the right line!
                        linenum--;
                    }
                    else
                    {
                        errline += " Failed too many times... aborting.";
                        ReportError(errline);
                        return;
                    }
                    //return;
                }
            }
            //            port.DataReceived += new SerialDataReceivedEventHandler(port_DataReceived);
        }

       
             private class Label
        {
            public String label;
            public int line;
            public Label(String _label, int _line){ label = _label; line = _line;}
        };

        public class SpecialCommandResult : CommandResult
        {
            public String       jumpTarget;

            public SpecialCommandResult() : base() { jumpTarget = ""; }
        };

        public class CommandResult
        {
            public enum CommandType { Success, Fail, Jump, JumpError, JumpSuccess, Halt }
            public      CommandType cmdType;

            public CommandResult()
            {   cmdType = CommandType.Success;   }
        };
    }
}
